<?php
// app/Services/BMKGService.php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;

class BMKGService
{
    protected $baseUrl = 'https://data.bmkg.go.id/DataMKG/MEWS/DigitalForecast/';
    
    /**
     * Get cuaca forecast untuk wilayah tertentu
     */
    public function getCuacaForecast($province = 'DIYogyakarta', $days = 7)
    {
        // Cache untuk 6 jam
        $cacheKey = "bmkg_forecast_{$province}_{$days}";
        
        return Cache::remember($cacheKey, now()->addHours(6), function () use ($province) {
            try {
                // BMKG API endpoint (disesuaikan dengan API resmi)
                $response = Http::timeout(10)->get($this->baseUrl . "DigitalForecast-{$province}.xml");
                
                if ($response->successful()) {
                    return $this->parseXMLData($response->body());
                }
                
                return $this->getFallbackData();
            } catch (\Exception $e) {
                \Log::error('BMKG API Error: ' . $e->getMessage());
                return $this->getFallbackData();
            }
        });
    }
    
    /**
     * Parse XML data dari BMKG
     */
    protected function parseXMLData($xmlString)
    {
        try {
            $xml = simplexml_load_string($xmlString);
            $data = [];
            
            foreach ($xml->forecast->area as $area) {
                $areaName = (string) $area['description'];
                $forecasts = [];
                
                foreach ($area->parameter as $param) {
                    $paramId = (string) $param['id'];
                    
                    if ($paramId === 'hu') { // Humidity
                        foreach ($param->timerange as $time) {
                            $forecasts[] = [
                                'datetime' => (string) $time['datetime'],
                                'humidity' => (int) $time->value[0],
                            ];
                        }
                    }
                }
                
                $data[$areaName] = $forecasts;
            }
            
            return $data;
        } catch (\Exception $e) {
            \Log::error('XML Parse Error: ' . $e->getMessage());
            return $this->getFallbackData();
        }
    }
    
    /**
     * Fallback data jika API gagal
     */
    protected function getFallbackData()
    {
        $forecasts = [];
        for ($i = 0; $i < 7; $i++) {
            $forecasts[] = [
                'tanggal' => now()->addDays($i)->format('Y-m-d'),
                'curah_hujan' => rand(0, 150),
                'kelembapan' => rand(60, 90),
                'suhu' => rand(24, 32),
            ];
        }
        return ['Gunungkidul' => $forecasts];
    }
    
    /**
     * Get curah hujan spesifik untuk desa
     */
    public function getCurahHujanDesa($desaId, $date = null)
    {
        $date = $date ?? now()->format('Y-m-d');
        $data = $this->getCuacaForecast();
        
        // Simulasi: ambil data random dari forecast
        // Dalam implementasi real, sesuaikan dengan koordinat desa
        $gunungkidulData = $data['Gunungkidul'] ?? [];
        
        foreach ($gunungkidulData as $forecast) {
            if ($forecast['tanggal'] === $date) {
                return $forecast;
            }
        }
        
        return [
            'tanggal' => $date,
            'curah_hujan' => rand(0, 100),
            'kelembapan' => rand(65, 85),
            'suhu' => rand(26, 31),
        ];
    }
}
